#pragma once
/*
 *  $Id: field.h 28963 2025-12-09 16:07:48Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef __GWY_FIELD_H__
#define __GWY_FIELD_H__

#include <libgwyddion/math.h>
#include <libgwyddion/interpolation.h>
#include <libgwyddion/line.h>
#include <libgwyddion/nield.h>

G_BEGIN_DECLS

#define GWY_TYPE_FIELD            (gwy_field_get_type())
#define GWY_FIELD(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), GWY_TYPE_FIELD, GwyField))
#define GWY_FIELD_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST((klass), GWY_TYPE_FIELD, GwyFieldClass))
#define GWY_IS_FIELD(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), GWY_TYPE_FIELD))
#define GWY_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), GWY_TYPE_FIELD))
#define GWY_FIELD_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), GWY_TYPE_FIELD, GwyFieldClass))

typedef struct _GwyField        GwyField;
typedef struct _GwyFieldClass   GwyFieldClass;
typedef struct _GwyFieldPrivate GwyFieldPrivate;

struct _GwyField {
    GObject parent_instance;
    GwyFieldPrivate *priv;

    gint xres;
    gint yres;
    gdouble xreal;
    gdouble yreal;
    gdouble xoff;
    gdouble yoff;
};

struct _GwyFieldClass {
    GObjectClass parent_class;

    void (*data_changed)(GwyField *field);

    /*< private >*/
    void (*reserved1)(void);
    void (*reserved2)(void);
};

GType           gwy_field_get_type           (void)                                G_GNUC_CONST;
GwyField*       gwy_field_new                (gint xres,
                                              gint yres,
                                              gdouble xreal,
                                              gdouble yreal,
                                              gboolean nullme)                     G_GNUC_MALLOC;
GwyField*       gwy_field_new_alike          (GwyField *model,
                                              gboolean nullme)                     G_GNUC_MALLOC;
GwyField*       gwy_field_copy               (GwyField *field)                     G_GNUC_MALLOC;
void            gwy_field_assign             (GwyField *destination,
                                              GwyField *source);
void            gwy_field_invalidate         (GwyField *field);
void            gwy_field_data_changed       (GwyField *field);
GwyField*       gwy_field_new_resampled      (GwyField *field,
                                              gint xres,
                                              gint yres,
                                              GwyInterpolationType interpolation);
void            gwy_field_resample           (GwyField *field,
                                              gint xres,
                                              gint yres,
                                              GwyInterpolationType interpolation);
void            gwy_field_bin                (GwyField *field,
                                              GwyField *target,
                                              gint binw,
                                              gint binh,
                                              gint xoff,
                                              gint yoff,
                                              gint trimlowest,
                                              gint trimhighest);
GwyField*       gwy_field_new_binned         (GwyField *field,
                                              gint binw,
                                              gint binh,
                                              gint xoff,
                                              gint yoff,
                                              gint trimlowest,
                                              gint trimhighest);
void            gwy_field_resize             (GwyField *field,
                                              gint xres,
                                              gint yres);
void            gwy_field_crop               (GwyField *field,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height);
GwyField*       gwy_field_area_extract       (GwyField *field,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height);
void            gwy_field_copy_data          (GwyField *field,
                                              GwyField *target);
void            gwy_field_area_copy          (GwyField *src,
                                              GwyField *dest,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height,
                                              gint destcol,
                                              gint destrow);
gdouble*        gwy_field_get_data           (GwyField *field);
const gdouble*  gwy_field_get_data_const     (GwyField *field);
gint            gwy_field_get_xres           (GwyField *field);
gint            gwy_field_get_yres           (GwyField *field);
gdouble         gwy_field_get_xreal          (GwyField *field);
gdouble         gwy_field_get_yreal          (GwyField *field);
void            gwy_field_set_xreal          (GwyField *field,
                                              gdouble xreal);
void            gwy_field_set_yreal          (GwyField *field,
                                              gdouble yreal);
gdouble         gwy_field_get_dx             (GwyField *field);
gdouble         gwy_field_get_dy             (GwyField *field);
gdouble         gwy_field_get_xoffset        (GwyField *field);
gdouble         gwy_field_get_yoffset        (GwyField *field);
void            gwy_field_set_xoffset        (GwyField *field,
                                              gdouble xoff);
void            gwy_field_set_yoffset        (GwyField *field,
                                              gdouble yoff);
GwyUnit*        gwy_field_get_unit_xy        (GwyField *field);
GwyUnit*        gwy_field_get_unit_z         (GwyField *field);
GwyValueFormat* gwy_field_get_value_format_xy(GwyField *field,
                                              GwyUnitFormatStyle style,
                                              GwyValueFormat *format);
GwyValueFormat* gwy_field_get_value_format_z (GwyField *field,
                                              GwyUnitFormatStyle style,
                                              GwyValueFormat *format);
void            gwy_field_copy_units         (GwyField *field,
                                              GwyField *target);
void            gwy_field_copy_units_to_line (GwyField *field,
                                              GwyLine *line);
void            gwy_line_copy_units_to_field (GwyLine *line,
                                              GwyField *field);
gdouble         gwy_field_itor               (GwyField *field,
                                              gdouble row);
gdouble         gwy_field_jtor               (GwyField *field,
                                              gdouble col);
gdouble         gwy_field_rtoi               (GwyField *field,
                                              gdouble realy);
gdouble         gwy_field_rtoj               (GwyField *field,
                                              gdouble realx);
gdouble         gwy_field_get_val            (GwyField *field,
                                              gint col,
                                              gint row);
void            gwy_field_set_val            (GwyField *field,
                                              gint col,
                                              gint row,
                                              gdouble value);
gdouble         gwy_field_get_dval           (GwyField *field,
                                              gdouble x,
                                              gdouble y,
                                              GwyInterpolationType interpolation);
gdouble         gwy_field_get_dval_real      (GwyField *field,
                                              gdouble x,
                                              gdouble y,
                                              GwyInterpolationType interpolation);
void            gwy_field_rotate             (GwyField *field,
                                              gdouble angle,
                                              GwyInterpolationType interpolation);
GwyField*       gwy_field_new_rotated        (GwyField *dfield,
                                              GwyField *exterior_mask,
                                              gdouble angle,
                                              GwyInterpolationType interp,
                                              GwyRotateResizeType resize);
GwyField*       gwy_field_new_rotated_90     (GwyField *field,
                                              gboolean clockwise);
void            gwy_field_flip               (GwyField *field,
                                              gboolean xflipped,
                                              gboolean yflipped);
void            gwy_field_invert_value       (GwyField *field);
void            gwy_field_transpose          (GwyField *src,
                                              GwyField *dest,
                                              gboolean minor);
void            gwy_field_area_transpose     (GwyField *src,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height,
                                              GwyField *dest,
                                              gboolean minor);
void            gwy_field_fill               (GwyField *field,
                                              gdouble value);
void            gwy_field_clear              (GwyField *field);
void            gwy_field_multiply           (GwyField *field,
                                              gdouble value);
void            gwy_field_add                (GwyField *field,
                                              gdouble value);
void            gwy_field_abs                (GwyField *field);
void            gwy_field_area_fill          (GwyField *field,
                                              GwyField *mask,
                                              GwyMaskingType mode,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height,
                                              gdouble value);
void            gwy_NIELD_area_fill          (GwyField *field,
                                              GwyNield *mask,
                                              GwyMaskingType mode,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height,
                                              gdouble value);
void            gwy_field_area_fill_mask     (GwyField *field,
                                              GwyNield *mask,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height,
                                              gdouble masked,
                                              gdouble unmasked);
void            gwy_field_area_clear         (GwyField *field,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height);
void            gwy_field_area_multiply      (GwyField *field,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height,
                                              gdouble value);
void            gwy_field_area_add           (GwyField *field,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height,
                                              gdouble value);
void            gwy_field_area_abs           (GwyField *field,
                                              gint col,
                                              gint row,
                                              gint width,
                                              gint height);
GwyLine*        gwy_field_get_profile        (GwyField *field,
                                              GwyLine* line,
                                              gint scol,
                                              gint srow,
                                              gint ecol,
                                              gint erow,
                                              gint res,
                                              gint thickness,
                                              GwyInterpolationType interpolation);
GwyXY*          gwy_field_get_profile_mask   (GwyField *field,
                                              gint *ndata,
                                              GwyField *mask,
                                              GwyMaskingType masking,
                                              gdouble xfrom,
                                              gdouble yfrom,
                                              gdouble xto,
                                              gdouble yto,
                                              gint res,
                                              gint thickness,
                                              GwyInterpolationType interpolation);
GwyXY*          gwy_NIELD_get_profile_mask   (GwyField *field,
                                              GwyNield *mask,
                                              GwyMaskingType masking,
                                              gdouble xfrom,
                                              gdouble yfrom,
                                              gdouble xto,
                                              gdouble yto,
                                              gint res,
                                              gint thickness,
                                              GwyInterpolationType interpolation,
                                              gint *ndata);
void            gwy_field_get_row            (GwyField *field,
                                              GwyLine* line,
                                              gint row);
void            gwy_field_get_column         (GwyField *field,
                                              GwyLine* line,
                                              gint col);
void            gwy_field_set_row            (GwyField *field,
                                              GwyLine* line,
                                              gint row);
void            gwy_field_set_column         (GwyField *field,
                                              GwyLine* line,
                                              gint col);
void            gwy_field_get_row_part       (GwyField *field,
                                              GwyLine* line,
                                              gint row,
                                              gint from,
                                              gint to);
void            gwy_field_get_column_part    (GwyField *field,
                                              GwyLine* line,
                                              gint col,
                                              gint from,
                                              gint to);
void            gwy_field_set_row_part       (GwyField *field,
                                              GwyLine* line,
                                              gint row,
                                              gint from,
                                              gint to);
void            gwy_field_set_column_part    (GwyField *field,
                                              GwyLine* line,
                                              gint col,
                                              gint from,
                                              gint to);
gdouble         gwy_field_get_xder           (GwyField *field,
                                              gint col,
                                              gint row);
gdouble         gwy_field_get_yder           (GwyField *field,
                                              gint col,
                                              gint row);
gdouble         gwy_field_get_angder         (GwyField *field,
                                              gint col,
                                              gint row,
                                              gdouble theta);
void            gwy_field_average_xyz        (GwyField *field,
                                              GwyField *density_map,
                                              const GwyXYZ *points,
                                              gint npoints);

G_END_DECLS

#endif

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
